/******************************************************************************/
/* OpenSi : Outils libres de gestion d'entreprise                             */
/* Copyright (C) 2003 Speedinfo.fr S.A.R.L.                                   */
/* Contact: contact@opensi.org                                                */
/*                                                                            */
/* This program is free software; you can redistribute it and/or              */
/* modify it under the terms of the GNU General Public License                */
/* as published by the Free Software Foundation; either version 2             */
/* of the License, or (at your option) any later version.                     */
/*                                                                            */
/* This program is distributed in the hope that it will be useful,            */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of             */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the               */
/* GNU General Public License for more details.                               */
/*                                                                            */
/* You should have received a copy of the GNU General Public License          */
/* along with this program; if not, write to the Free Software                */
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */
/******************************************************************************/

package org.opensi.facturation;


import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import org.experlog.openeas.api.Session;
import org.opensi.util.tools.DateTime;


public class StockManager {
	
	private Connection con;
	private String base;
	
	private Session s;	

	private PreparedStatement psMvtE;
	private PreparedStatement psMvtS;
	private PreparedStatement psStkI;
	private PreparedStatement psStkE;
	private PreparedStatement psStkS;
	private PreparedStatement psStkCC;
	private PreparedStatement psStkCF;
	private PreparedStatement psCump; // dernier cump
	private PreparedStatement psSAN;  // mise  jour du stock  nouveau
	private PreparedStatement psEP;   // entres d'une priode
	private PreparedStatement psSP;   // sorties priode
	private PreparedStatement psVPP;  // valorisation priode prcdente
	private PreparedStatement psNomenclature;
	private PreparedStatement psMvtCC;
	private PreparedStatement psMvtCF;
	

	public StockManager(String s_) {
		try {
	
			s = Session.findClient(s_);
			con = s.getConnection(null);
			base = s.getCookie().get("BaseDossier");
			init();
			
		} catch(Exception e) {
			e.printStackTrace();
		}
	}
	
	
	public StockManager(Connection con, String base) {
		try {
	
			this.con = con;
			this.base = base;
			init();
			
		} catch(Exception e) {
			e.printStackTrace();
		}
	}
	
	
	private void init() {
		try {

			long dateMvt = (new DateTime()).getTimeInMillis();

			psNomenclature = con.prepareStatement("select ArticleComp_Id, Quantite from "+ base +".FICHE_ARTICLE a, "+ base +".COMPOSANT_ARTICLE c where a.Article_Id=c.Article_Id and a.Composition='N' and a.Article_Id=?");

			psMvtE = con.prepareStatement("insert into "+ base +".MVT_STOCK (Article_Id, Date_Mvt, Type_Mvt, Quantite, Num_Piece, Libelle, Prix_HT, Frais_Appro) values (?,"+ dateMvt +",'E',?,?,?,?,?)");
			psMvtS = con.prepareStatement("insert into "+ base +".MVT_STOCK (Article_Id, Date_Mvt, Type_Mvt, Quantite, Num_Piece, Libelle) values (?,"+ dateMvt +",'S',?,?,?)");

			psStkI = con.prepareStatement("update "+ base +".STOCKS_ARTICLE set Stock_Init=? where Article_Id=?");
			psStkE = con.prepareStatement("update "+ base +".STOCKS_ARTICLE set Entrees=Entrees+? where Article_Id=?");
			psStkS = con.prepareStatement("update "+ base +".STOCKS_ARTICLE set Sorties=Sorties+? where Article_Id=?");
			psStkCC = con.prepareStatement("update "+ base +".STOCKS_ARTICLE set Com_Clients=Com_Clients+? where Article_Id=?");
			psStkCF = con.prepareStatement("update "+ base +".STOCKS_ARTICLE set Com_Fournisseurs=Com_Fournisseurs+? where Article_Id=?");
			
			psMvtCC = con.prepareStatement("insert into "+ base +".MVT_COM_CLIENT (Article_Id, Date_Mvt, Quantite, Num_Piece, Libelle) select Fiche_Article_Id,"+ dateMvt +",?,?,? from "+ base +".FICHE_ARTICLE where Article_Id=?");
			psMvtCF = con.prepareStatement("insert into "+ base +".MVT_COM_FOURNISSEUR (Article_Id, Date_Mvt, Quantite, Num_Piece, Libelle) select Fiche_Article_Id,"+ dateMvt +",?,?,? from "+ base +".FICHE_ARTICLE where Article_Id=?");
			
			psCump = con.prepareStatement("select s.Valorisation, s.Frais_Appro from "+ base +".STOCK_ANOUVEAU s, "+ base +".FICHE_ARTICLE f where f.Article_Id=? and s.Article_Id=f.Fiche_Article_Id order by Periode desc limit 1");
		
			psEP = con.prepareStatement("select sum(Quantite), sum(Quantite * Prix_HT), sum(Quantite * Frais_Appro) from "+ base +".MVT_STOCK where Type_Mvt='E' and Article_Id=? and Date_Mvt>=? and Date_Mvt<?");
			psSP = con.prepareStatement("select sum(Quantite) from "+ base +".MVT_STOCK where Type_Mvt='S' and Article_Id=? and Date_Mvt>=? and Date_Mvt<?");
		
			psSAN = con.prepareStatement("insert into "+ base +".STOCK_ANOUVEAU (Article_Id, Periode, Valorisation, Quantite, Frais_Appro) values (?,?,?,?,?) on duplicate key update Valorisation=?, Quantite=?, Frais_Appro=?");
			
			psVPP = con.prepareStatement("select f.Fiche_Article_Id, coalesce(s.Quantite,0), coalesce(s.Valorisation,0), coalesce(s.Frais_Appro,0) from "+ base +".FICHE_ARTICLE f left join "+ base +".STOCK_ANOUVEAU s on s.Article_Id=f.Fiche_Article_Id and Periode<? where f.Article_Id=? order by Periode desc limit 1");

		} catch(Exception e) {
			e.printStackTrace();
		}
	}
	
	
	public synchronized void initStock(String articleId, double qte, String libelle, double prix, double frais) throws SQLException {
		psStkI.setDouble(1, qte);
		psStkI.setString(2, articleId);
		psStkI.executeUpdate();
		
		psMvtE.setString(1, articleId);
		psMvtE.setDouble(2, qte);
		psMvtE.setString(3, "-");
		psMvtE.setString(4, libelle);
		psMvtE.setDouble(5, prix);
		psMvtE.setDouble(6, frais);
		psMvtE.executeUpdate();
		
		updateSAN(articleId);
	}
	
	
	public synchronized void incEntrees(String articleId, double qte, String numPiece, String libelle, double prix, double frais) throws SQLException {
		psStkE.setDouble(1, qte);
		psStkE.setString(2, articleId);
		psStkE.executeUpdate();
		
		psMvtE.setString(1, articleId);
		psMvtE.setDouble(2, qte);
		psMvtE.setString(3, numPiece);
		psMvtE.setString(4, libelle);
		psMvtE.setDouble(5, prix);
		psMvtE.setDouble(6, frais);
		psMvtE.executeUpdate();
		
		updateSAN(articleId);
	}
	
	
	public synchronized void incEntrees(String articleId, double qte, String numPiece, String libelle) throws SQLException {
		
		PreparedStatement psForfaits = con.prepareStatement("select ArticleComp_Id, Quantite from "+ base +".FICHE_ARTICLE a, "+ base +".COMPOSANT_ARTICLE c where a.Article_Id=c.Article_Id and a.Composition='F' and a.Article_Id=?");
		psForfaits.setString(1, articleId);
		ResultSet rset = psForfaits.executeQuery();

		if (rset.next()) {
			do {			
				incEntrees(rset.getString("ArticleComp_Id"), qte * rset.getDouble("Quantite"), numPiece, libelle);
			}
			while (rset.next());
		}
		else {		
			psCump.setString(1, articleId);		
			ResultSet rsCump = psCump.executeQuery();

			psStkE.setDouble(1, qte);
			psStkE.setString(2, articleId);
			psStkE.executeUpdate();

			psMvtE.setString(1, articleId);
			psMvtE.setDouble(2, qte);
			psMvtE.setString(3, numPiece);
			psMvtE.setString(4, libelle);
			if (rsCump.next()) {
				double valorisation = rsCump.getDouble("Valorisation");
				double fraisAppro = rsCump.getDouble("Frais_Appro");
				psMvtE.setDouble(5, (valorisation>0?valorisation:0));
				psMvtE.setDouble(6, (fraisAppro>0?fraisAppro:0));
			}
			else {
				psMvtE.setDouble(5, 0);
				psMvtE.setDouble(6, 0);				
			}

			psMvtE.executeUpdate();
		}

		updateSAN(articleId);
	}
	
	public void assembler(String articleId, double qte, String numPiece, String libelle) throws SQLException {
		
		incEntrees(articleId, qte, numPiece, libelle);
		
		psNomenclature.setString(1, articleId);
		ResultSet rset = psNomenclature.executeQuery();

		while (rset.next()) {
			incSorties(rset.getString("ArticleComp_Id"), qte * rset.getDouble("Quantite"), numPiece, libelle);
		}
	}	
	
	public void desassembler(String articleId, double qte, String numPiece, String libelle) throws SQLException {
		
		incSorties(articleId, qte, numPiece, libelle);
		
		psNomenclature.setString(1, articleId);
		ResultSet rset = psNomenclature.executeQuery();

		while (rset.next()) {
			incEntrees(rset.getString("ArticleComp_Id"), qte * rset.getDouble("Quantite"), numPiece, libelle);
		}
	}	
	
	
	public synchronized void incSorties(String articleId, double qte, String numPiece, String libelle) {
		try {
			
			PreparedStatement psForfaits = con.prepareStatement("select ArticleComp_Id, Quantite from "+ base +".FICHE_ARTICLE a, "+ base +".COMPOSANT_ARTICLE c where a.Article_Id=c.Article_Id and a.Composition='F' and a.Article_Id=?");
			psForfaits.setString(1, articleId);
			ResultSet rset = psForfaits.executeQuery();

			if (rset.next()) {
				do {			
					incSorties(rset.getString("ArticleComp_Id"), qte * rset.getDouble("Quantite"), numPiece, libelle);
				}
				while (rset.next());
			}
			else {
				psStkS.setDouble(1, qte);
				psStkS.setString(2, articleId);
				psStkS.executeUpdate();

				psMvtS.setString(1, articleId);
				psMvtS.setDouble(2, qte);
				psMvtS.setString(3, numPiece);
				psMvtS.setString(4, libelle);
				psMvtS.executeUpdate();
			}
			updateSAN(articleId);
		
		} catch(SQLException e) {
			e.printStackTrace();
		}
	}
	
	
	public synchronized void incComFournisseurs(String articleId, double qte, String numPiece, String libelle) throws SQLException {
		updateComFournisseurs(articleId, qte, numPiece, libelle);
	}
	
	public synchronized void decComFournisseurs(String articleId, double qte, String numPiece, String libelle) throws SQLException {
		updateComFournisseurs(articleId, -qte, numPiece, libelle);
	}
	
	private void updateComFournisseurs(String articleId, double qte, String numPiece, String libelle) throws SQLException {
		psStkCF.setDouble(1, qte);
		psStkCF.setString(2, articleId);
		psStkCF.executeUpdate();

		psMvtCF.setDouble(1, qte);
		psMvtCF.setString(2, numPiece);
		psMvtCF.setString(3, libelle);
		psMvtCF.setString(4, articleId);
		psMvtCF.executeUpdate();
	}
		
	public synchronized void incComClients(String articleId, double qte, String numPiece, String libelle) throws SQLException {
		updateComClients(articleId, qte, numPiece, libelle);
	}
	
	
	public synchronized void decComClients(String articleId, double qte, String numPiece, String libelle) throws SQLException {
		updateComClients(articleId, -qte, numPiece, libelle);
	}
	
	
	private void updateComClients(String articleId, double qte, String numPiece, String libelle) throws SQLException {
	
		PreparedStatement psForfaits = con.prepareStatement("select ArticleComp_Id, Quantite from "+ base +".FICHE_ARTICLE a, "+ base +".COMPOSANT_ARTICLE c where a.Article_Id=c.Article_Id and a.Composition='F' and a.Article_Id=?");
		psForfaits.setString(1, articleId);
		ResultSet rset = psForfaits.executeQuery();
		
		if (rset.next()) {		
			do {			
				updateComClients(rset.getString("ArticleComp_Id"), qte * rset.getDouble("Quantite"), numPiece, libelle);
			}
			while (rset.next());
		}
		else {
			psStkCC.setDouble(1, qte);
			psStkCC.setString(2, articleId);
			psStkCC.executeUpdate();
			
			psMvtCC.setDouble(1, qte);
			psMvtCC.setString(2, numPiece);
			psMvtCC.setString(3, libelle);
			psMvtCC.setString(4, articleId);
			psMvtCC.executeUpdate();			
		}
	}
	
	
	private void updateSAN(String reference) throws SQLException {
		
		DateTime dt = new DateTime();
		dt.setDay(1);
		long debutPeriode = dt.getDateInMillis();
		dt.setMonth(dt.getMonth()+1);		
		long finPeriode = dt.getDateInMillis();
		long periode = finPeriode;
		
		psVPP.setLong(1, finPeriode);
		psVPP.setString(2, reference);
		ResultSet rsVPP = psVPP.executeQuery();

		rsVPP.next();
		
		int articleId = rsVPP.getInt(1);
		double stockPrecedent = rsVPP.getDouble(2);
		double valorisationPrecedente = rsVPP.getDouble(3);		
		double fraisPrecedent = rsVPP.getDouble(4);

		rsVPP.close();

		psEP.setString(1, reference);
		psEP.setLong(2, debutPeriode);
		psEP.setLong(3, finPeriode);
		
		ResultSet rsEP = psEP.executeQuery();				

		rsEP.next();

		double entrees = rsEP.getDouble(1);
		double stockPrecedentValo = (stockPrecedent<0)?0:stockPrecedent;
		
		double valorisation = (entrees + stockPrecedentValo>0)?(rsEP.getDouble(2) + valorisationPrecedente * stockPrecedentValo) / (entrees + stockPrecedentValo):valorisationPrecedente;

		double fraisAppro = (entrees + stockPrecedentValo>0)?(rsEP.getDouble(3) + fraisPrecedent * stockPrecedentValo) / (entrees + stockPrecedentValo):fraisPrecedent;
		
		rsEP.close();
		
		psSP.setString(1, reference);
		psSP.setLong(2, debutPeriode);
		psSP.setLong(3, finPeriode);

		ResultSet rsSP = psSP.executeQuery();

		rsSP.next();

		double sorties = rsSP.getDouble(1);
		
		rsSP.close();
		
		double quantite = stockPrecedent + entrees - sorties;
		
		psSAN.setInt(1, articleId);
		psSAN.setLong(2, periode);
		psSAN.setDouble(3, valorisation);
		psSAN.setDouble(4, quantite);
		psSAN.setDouble(5, fraisAppro);
		psSAN.setDouble(6, valorisation);
		psSAN.setDouble(7, quantite);
		psSAN.setDouble(8, fraisAppro);
		psSAN.executeUpdate();
	}
	
	protected void finalize() {
		try {
			if (s!=null) {
				s.closeConnection(con, null);
			}
		}
		catch(SQLException e) {
			e.printStackTrace();
		}
	}	
	
} // fin StockManager

